/**
 * jQuery Hotspot : A jQuery Plugin to create hotspot for an HTML element
 *
 * Author: Aniruddha Nath
 * Version: 1.0.0
 * 
 * Website: https://github.com/Aniruddha22/jquery-hotspot
 * 
 * Description: A jquery plugin for creating and displaying Hotspots in an HTML element.
 *	It operates in two mode, admin and display. The design of the hotspot created are fully customizable.
 *	User can add their own CSS class to style the hotspots.
 * 
 * License: http://www.opensource.org/licenses/mit-license.php
 */

;(function($) {
	
	// Default settings for the plugin
	var defaults = {

		// Data
		data: [],

		// Hotspot Tag
		tag: 'img',

		// Mode of the plugin
		// Options: admin, display
		mode: 'display',

		// HTML5 LocalStorage variable
		LS_Variable: '__HotspotPlugin_LocalStorage',

		// Hidden class for hiding the data
		hiddenClass: 'hidden',

		// Event on which the data will show up
		// Options: click, hover, none
		interactivity: 'hover',

		// Buttons' id (Used only in Admin mode)
		done_btnId: 'HotspotPlugin_Done',
		remove_btnId: 'HotspotPlugin_Remove',
		sync_btnId: 'HotspotPlugin_Server',

		// Buttons class
		done_btnClass: 'btn btn-success HotspotPlugin_Done',
		remove_btnClass: 'btn btn-danger HotspotPlugin_Remove',
		sync_btnClass: 'btn btn-info HotspotPlugin_Server',

		// Classes for Hotspots
		hotspotClass: 'HotspotPlugin_Hotspot',
		hotspotAuxClass: 'HotspotPlugin_inc',

		// Overlay
		hotspotOverlayClass: 'HotspotPlugin_Overlay',

		// Enable ajax
		ajax: false,

		ajaxOptions: {
			url: ''
		},

		// No. of variables included in the spots
		dataStuff: [
			{
				'titl': 'Marker Text',
				'property': 'text',
				'htype': 'text',
				'default': ''
			},
			{
				'titl': 'Marker Shape',
				'property': 'shape',
				'htype': 'dropdown',
				'class': 'three-block',
				'default': 'round',
				'choices': { 'Round': 'round', 'Rounded': 'rounded', 'Square': 'square'}
			},
			{
				'titl': 'Marker Size',
				'property': 'size',
				'htype': 'dropdown',
				'class': 'three-block',
				'default': 'small',
				'choices': { 'Small': 'small', 'Normal': 'normal', 'Large': 'large', 'X-Large': 'x-large'}
			},
			{
				'titl': 'Marker Animation Style',
				'property': 'style',
				'htype': 'dropdown',
				'class': 'three-block',
				'default': '',
				'choices': { 'None': '', 'Animation 1': '1', 'Animation 2': '2', 'Animation 3': '3'}
			},
			{
				'titl': 'Tooltip Position',
				'property': 'position',
				'htype': 'dropdown',
				'class': 'three-block',
				'default': 'right',
				'choices': { 'Right': 'right', 'Left': 'left', 'Top': 'top', 'Bottom': 'bottom'}
			},
			{
				'titl': 'Color',
				'property': 'Color',
				'class': 'three-block',
				'htype': 'color',
				'default': '#fff'
			},
			{
				'titl': 'Background Color',
				'property': 'bgcolor',
				'class': 'three-block',
				'htype': 'color',
				'default': '#e7512f'
			},
			{
				'titl': 'Image',
				'property': 'image',
				'htype': 'image',
				'default': ''
			},
			{
				'titl': 'Title',
				'property': 'Title',
				'htype': 'text',
				'default': ''
			},
			{
				'titl': 'Content',
				'property': 'Message',
				'htype': 'textarea',
				'default': ''
			}
		]
	};

	count = 1;

	//Constructor
	function Hotspot(element, options) {
		
		// Overwriting defaults with options
		this.config = $.extend(true, {}, defaults, options);
		
		this.element = element;
		this.imageEl = element.find(this.config.tag);
		this.imageParent = this.imageEl.parent();

		this.broadcast = '';

		var widget = this;

		// Event API for users
		$.each(this.config, function(index, val) {
			if (typeof val === 'function') {
				widget.element.on(index + '.hotspot', function() {
					val(widget.broadcast);
				});
			};
		});

		this.init();
	}

	Hotspot.prototype.init = function() {

		this.getData();
		if (this.config.mode != 'admin') {
			return;
		};
		var mainImage 	= new Image(),
			self 		= this;
		
		mainImage.src 	= $(this.imageEl).attr('src');
		mainImage.onload = function() {
			var height 	= self.imageEl[0].height;
			var width 	= self.imageEl[0].width;

			// Masking the image
			$('<span/>', {html: ''}).addClass(self.config.hotspotOverlayClass).appendTo(self.imageParent);

			var widget 	= self,
				data 	= [],
				marker = count;

			// Adding controls
			$('<button/>', {html: '<i title="Remove" class="fa fa-times"></i> Remove All'}).prop('id', self.config.remove_btnId).addClass(self.config.remove_btnClass).appendTo(self.imageParent);

			$(self.imageParent).delegate('button#' + self.config.remove_btnId, 'click', function(event) {
				event.preventDefault();
				widget.removeData();
			});

			// Start storing
			self.element.delegate('span', 'click', function(event) {
				event.preventDefault();
				marker++;
				var offset 		= $(this).offset(),
					relX 		= (event.pageX - offset.left) / width * 100,
					relY 		= (event.pageY - offset.top) / height * 100,
					dataStuff 	= widget.config.dataStuff,
					dataBuild 	= {index: marker, x: relX, y: relY};

				for (var i = 0; i < dataStuff.length; i++) {
					var val = dataStuff[i];
					dataBuild[val.property] = val.default;
				};

				data.push(dataBuild);

				if (widget.config.interactivity === 'none') {
					var htmlBuilt = $('<div data-index="'+marker+'"><i title="Remove" class="remove-mark fa fa-times"></i></div>');
				} else {
					var htmlBuilt = $('<div data-index="'+marker+'"><i title="Remove" class="remove-mark fa fa-times"></i></div>').addClass(widget.config.hiddenClass);
				}


				$.each(dataBuild, function(index, val) {
					if (typeof val === "string") {
						$('<div/>', {html: val}).addClass('Hotspot_' + index).appendTo(htmlBuilt);
					};
				});

				var div = $('<div/>', {html: htmlBuilt}).css({
					'top': relY + '%',
					'left': relX + '%'
				}).addClass(widget.config.hotspotClass + ' ' + widget.config.hotspotAuxClass).appendTo(widget.element);

				if (widget.config.interactivity === 'click') {
					div.on(widget.config.interactivity, function(event) {
						$(this).children('div').toggleClass(widget.config.hiddenClass);
					});
					htmlBuilt.css('display', 'block');
				} else {
					htmlBuilt.removeClass(widget.config.hiddenClass);
				}

				if (widget.config.interactivity === 'none') {
					htmlBuilt.css('display', 'block');
				};
				
				widget.storeData(data);
				data = [];
				
				widget.editModal(marker);

				$('.color-chooser').wpColorPicker();
				self.insertImage();
				
			});
			
			self.element.delegate('.HotspotPlugin_Hotspot', 'click', function(e) {
				e.preventDefault();
				var $self = $(this),
					index = $self.find('> div').data('index'),
					newdata = widget.getMarkData(index)[0];
				
				widget.editModal(index,newdata);
				$('.color-chooser').wpColorPicker();
				self.insertImage();
			});
			
			self.element.delegate('.remove-mark', 'click', function(e) {
				e.preventDefault();
				e.stopPropagation();
				var index = $(this).parent().data('index');
				widget.removeMark(index);
				setTimeout(function() {
					widget.updatePositions();
				}, 0);
			});
			
			if(typeof $.fn.draggable != 'undefined') {
				var config = {
					containment: 'parent',
					stop: function(event, ui) {
						var index = +$(event.target).find('> div').data('index'),
							data = {};

						data.x = ui.position.left / width * 100;
						data.y = ui.position.top / height * 100;

						widget.updateData(data, index);
					}
				};
				if($('.HotspotPlugin_Hotspot').length > 0) {
					$('.HotspotPlugin_Hotspot').draggable(config);
					$('body').on('cp-hotspot-updated', function() {
						$('.HotspotPlugin_Hotspot:not(.ui-draggable-handle)').draggable(config);
					});
				}
			}
		};
	};

	Hotspot.prototype.editModal = function(index, currentData) {
		
		var self 			= this,
			dataStuff 		= this.config.dataStuff,
			modalHtml 		= '';
		
		for (var i = 0; i < dataStuff.length; i++) {
			var val 		= dataStuff[i],
				defaultText = (typeof currentData != 'undefined' && typeof currentData[val.property] != 'undefined') ? currentData[val.property] : val.default,
				cls 		= ( typeof val.class != 'undefined') ? ' ' + val.class : '';
			
			if(val.htype == 'text') {
				input = '<label>'+val.titl+'</label><input type="text" class="cp-hotspot-'+val.property+'" value="'+defaultText+'" />';
			}

			if(val.htype == 'textarea') {
				input = '<label>'+val.titl+'</label><textarea class="cp-hotspot-'+val.property+'">'+defaultText+'</textarea>';
			}

			if(val.htype == 'dropdown') {
				input = '<label>'+val.titl+'</label>'+ self.createSelect( val, defaultText );
			}

			if(val.htype == 'image') {
				input = '<label>'+val.titl+'</label><input class="hidden custom-up cp-hotspot-'+val.property+'" type="text" value="' + defaultText + '" /><a class="upload_image_hot" href="#"></a>';
                input += '<span class="clear-img"><img class="logo-im" alt="Upload" src="" /><a class="remove-img" href="#" title="Remove Image"></a></span>';
			}

			if(val.htype == 'color') {
				input = '<label>'+val.titl+'</label><input class="color-chooser cp-hotspot-'+val.property+'" type="text" data-alpha="true"  data-default-color="' + defaultText + '" value="' + defaultText + '" />';
			}
			
			modalHtml += '<div class="cp-hotspot-block'+cls+'">'+input+'</div>';
		};
		
		modalHtml += '<div class="cp-modal-footer"><a href="#" class="cp-hotspot-button cp-hotspot-save">Save Changes</a><a href="#" class="cp-hotspot-button cp-hotspot-close">Cancel</a></div>';
		$('body').append('<div class="cp-hotspot-popup"><div class="cp-hotspot-popup-title">Hotspot Tooltip Content<a href="#" title="Close" class="cp-hotspot-close cp-close-modal fa fa-times"></a></div>'+modalHtml+'</div>');
		

		$('.cp-hotspot-save').on('click', function() {
			var wrap = $(this).parents('.cp-hotspot-popup'),
				dataBuild = {};
			
			for (var i = 0; i < dataStuff.length; i++) {
				var val = dataStuff[i];
				dataBuild[val.property] = wrap.find('.cp-hotspot-'+val.property).val();
			};
			
			$('.cp-hotspot-popup').remove();
			self.updateData(dataBuild, index);
			self.updatePositions();
		});
		
		$('body').on('click', '.cp-hotspot-close', function() {
			$('.cp-hotspot-popup').remove();
		});

	};

	Hotspot.prototype.insertImage = function() {
		$('.upload_image_hot').each(function(){
			var parnt = $(this).parent(),
				txtb = $(this).prev();

			if(txtb.val() != ''){
				parnt.find('.clear-img').show(0);
				parnt.find('.logo-im').attr('src',txtb.val());
			}



			$(this).on('click', function(e) {
				e.preventDefault();
				wp.media.editor.send.attachment = function(props, attachment) {
					txtb.val(attachment.url);
					
					if( parnt.find('.logo-im').length == 0 ) {
						parnt.find('.upload_image_button').after("<span class='clear-img'><img class='logo-im' alt='' src='"+attachment.url+"' /><a href='#' class='remove-img' style='display:inline-block'></a></span>");
					}

					parnt.find('img.logo-im').attr('src',attachment.url);
					parnt.find('.clear-img').show(0);
					parnt.find('.remove-img').show();			    	

			    	parnt.find('.remove-img').on('click', function(e){
						e.preventDefault();
						parnt.find('.clear-img').remove();
						parnt.find('.cp-hotspot-image').val('');
				    	
					});

				}
				wp.media.editor.open(this);
				return false;
			});
		});

		$('.remove-img').each(function(){
			$(this).on('click', function(e){
				e.preventDefault();
				$(this).parent().parent().find('.cp-hotspot-image').val('');
				$(this).parent().remove();
			});
		});
		
	};

	Hotspot.prototype.createSelect = function( val, defaultText ) {
		var $select = $('<select></select>');
	    var $option;

	    for (var vals in val.choices) {
	        $option = $('<option value="' + val.choices[vals] + '">' + vals + '</option>');
	        if (val.choices[vals] == defaultText) {
	            $option.attr('selected', 'selected');
	        }
	        $select.append( $option );
	    }
	    return '<select class="cp-hotspot-'+val.property+'">' + $select.html() + '</select>';
	};

	Hotspot.prototype.getMarkData = function(index) {
		if (index == '') {
			return;
		};

		var raw = decodeURIComponent($(this.config.LS_Variable).val()),
			obj = [],
			mark = [];
		
		if (raw) {
			obj = JSON.parse(raw);
		}

		$.each(obj, function(count) {
			var elem = obj[count];
			
			if(elem['index'] == index) {
				mark.push(elem);
			}
		});
		
		return mark;
	};

	Hotspot.prototype.getData = function() {
		var widget = this;
		
		if (($(this.config.LS_Variable).val() == '' || $(this.config.LS_Variable).val()) === null && this.config.data.length == 0) {
			return;			
		} 

		if (this.config.mode == 'admin' && ($(this.config.LS_Variable).val() == '' || $(this.config.LS_Variable).val() === null)) {
			return;
		} 
		
		this.beautifyData();
	}

	Hotspot.prototype.beautifyData = function() {
		var widget = this;

		if (this.config.mode != 'admin' && this.config.data.length != 0) {
			var obj = this.config.data;
		} else {
			var raw = decodeURIComponent($(this.config.LS_Variable).val());
			var obj = JSON.parse(raw);
		}
		
		for (var i = obj.length - 1; i >= 0; i--) {
			var el = obj[i];
			
			if(i == obj.length - 1) {
				count = el['index'];
			}

			if (this.config.interactivity === 'none') {
				var htmlBuilt = $('<div data-index="'+el.index+'"><i title="Remove" class="remove-mark fa fa-times"></i></div>');
			} else {
				var htmlBuilt = $('<div data-index="'+el.index+'"><i title="Remove" class="remove-mark fa fa-times"></i></div>').addClass(this.config.hiddenClass);
			}

			$.each(el, function(index, val) {
				if (typeof val === "string") {
					$('<div/>', {html: val}).addClass('Hotspot_' + index).appendTo(htmlBuilt);
				};
			});

			var div = $('<div/>', {html: htmlBuilt}).css({
				'top': el.y + '%',
				'left': el.x + '%'
			}).addClass(this.config.hotspotClass).appendTo(this.element);

			if (widget.config.interactivity === 'click') {
				div.on(widget.config.interactivity, function(event) {
					$(this).children('div').toggleClass(widget.config.hiddenClass);
				});
				htmlBuilt.css('display', 'block');
			} else {
				htmlBuilt.removeClass(this.config.hiddenClass);
			}

			htmlBuilt.find('.Hotspot_text,.Hotspot_style,.Hotspot_shape,.Hotspot_size,.Hotspot_Color,.Hotspot_bgcolor,.Hotspot_image,.Hotspot_position').remove();

			if (this.config.interactivity === 'none') {
				htmlBuilt.css('display', 'block');
			}
		};
	};

	Hotspot.prototype.storeData = function(data) {

		if (data.length == 0) {
			return;
		};

		var raw = decodeURIComponent($(this.config.LS_Variable).val());
		obj = [];
		
		if (raw) {
			var obj = JSON.parse(raw);
		};

		$.each(data, function(index) {
			var node = data[index];

			obj.push(node);
		});

		$(this.config.LS_Variable).val(encodeURIComponent(JSON.stringify(obj)));

		this.broadcast = 'Saved to LocalStorage';
		this.element.trigger('afterSave.hotspot');
	};

	Hotspot.prototype.removeMark = function(index) {
		if (index == '') {
			return;
		};

		var raw = decodeURIComponent($(this.config.LS_Variable).val()),
			obj = [],
			mark = [];
		
		if (raw) {
			obj = JSON.parse(raw);
		}
		
		$.each(obj, function(count) {
			var elem = obj[count];
			
			if(elem['index'] != index) {
				mark.push(elem);
			}
		});
		
		$(this.config.LS_Variable).val(encodeURIComponent(JSON.stringify(mark)));

		this.broadcast = 'Saved to LocalStorage';
		this.element.trigger('afterSave.hotspot');
	};

	Hotspot.prototype.removeData = function() {
		if ($(this.config.LS_Variable).val === null) {
			return;
		};
		if (!confirm("Are you sure you wanna do everything?")) {
			return;
		};
		$(this.config.LS_Variable).val('');
		this.broadcast = 'Removed successfully';
		this.element.trigger('afterRemove.hotspot');
		this.updatePositions();
	};

	Hotspot.prototype.updateData = function(data, index) {

		if (data.length == 0 || index == '') {
			return;
		};

		var raw = decodeURIComponent($(this.config.LS_Variable).val()),
			obj = [];
		
		if (raw) {
			obj = JSON.parse(raw);
		}

		$.each(obj, function(count) {
			if(obj[count]['index'] == index) {
				$.each(obj[count], function(i) {
					if(typeof data[i] != 'undefined' && typeof obj[count][i] != 'undefined') {
						obj[count][i] = data[i];
					}
				});
			}
		});

		$(this.config.LS_Variable).val(encodeURIComponent(JSON.stringify(obj)));

		this.broadcast = 'Saved to LocalStorage';
		this.element.trigger('afterSave.hotspot');
	};

	Hotspot.prototype.updatePositions = function() {
		
		if(this.element.find('.HotspotPlugin_Hotspot').length > 0) {
			this.element.find('.HotspotPlugin_Hotspot').remove();
		}
		
		this.beautifyData();
		
	};

	$.fn.hotspot = function (options) {
		new Hotspot(this, options);
		return this;
	}

}(jQuery));